// $Id: CPresetBrowser.cpp,v 1.7 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CPresetBrowser.hpp"
#include "../Basics/CDialog.hpp"

//  ===========================================================================

using Exponent::GUI::Controls::CPresetBrowser;
using Exponent::GUI::Basics::CDialog;

//  ===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CPresetBrowser, CControlPanel);

//  ===========================================================================
CPresetBrowser::CPresetBrowser(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener) 
			  : CControlPanel(root->getParentWindow(), root, uniqueId, area, listener)
{
	EXPONENT_CLASS_CONSTRUCTION(CPresetBrowser);
    NULL_POINTER(m_presetHandler);
}

//  ===========================================================================
CPresetBrowser::~CPresetBrowser()
{
	EXPONENT_CLASS_DESTRUCTION(CPresetBrowser);
    FORGET_COUNTED_OBJECT(m_presetHead);
	FORGET_COUNTED_OBJECT(m_selector);
}

//  ===========================================================================
void CPresetBrowser::initialise(const SPresetBrowserSetup &setup)
{
	// Store the root path of the presets
	m_rootPath = setup.m_rootPath;

	// Create the selector
	m_selector = new CPresetBrowserSelector(this, 0, CRect(0, setup.m_headArea.getHeight(), m_area.getWidth(), m_area.getHeight() - setup.m_headArea.getHeight()), setup.m_theFont, NULL, m_presetHandler);

	// Create the head
	m_presetHead = new CPresetBrowserHead(this, 0, CRect(0, 0, setup.m_headArea.getWidth(), setup.m_headArea.getHeight()), this, m_presetHandler, m_selector);

	// Reference controls
	m_presetHead->referenced();
	m_selector->referenced();

	// Add the contorl
	this->addControl(m_presetHead);
	this->addControl(m_selector);

	// Now we want to initialise the controls, head first
	CPresetBrowserHead::SPresetBrowserHeadSetup headSetup;
//	headSetup.m_rootPath				= setup.m_rootPath;
	headSetup.m_rootPath.setString(setup.m_rootPath.getString());
	headSetup.m_bankComboRect			= setup.m_bankComboArea;
	headSetup.m_bankComboDownImage		= setup.m_bankComboDownImage;
	headSetup.m_bankComboUpImage		= setup.m_bankComboUpImage;
	headSetup.m_loadButtonDownImage		= setup.m_loadButtonDownImage;
	headSetup.m_loadButtonUpImage		= setup.m_loadButtonUpImage;
	headSetup.m_loadButtonRect			= setup.m_loadButtonArea;
	headSetup.m_saveButtonDownImage		= setup.m_saveButtonDownImage;
	headSetup.m_saveButtonUpImage		= setup.m_saveButtonUpImage;
	headSetup.m_saveButtonRect			= setup.m_saveButtonArea;
	headSetup.m_textColour				= setup.m_textColour;
	headSetup.m_theFont					= setup.m_theFont;

	// Initialise the head
	m_presetHead->intialise(headSetup);

	// Set the text colour for the selector
	m_selector->setFontColour(setup.m_textColour);

	// Get the files to add
	TCountedPointerArray<CFileInformation> files;
	CFileManager::getFilesInDirectory(files, setup.m_rootPath);

	// Loop through the files
	for (long i = 0; i < files.getArraySize(); i++)
	{
		// GEt the file info
		CFileInformation *info = files.elementAtIndex(i);

		if (info)
		{
			CTrace::trace("We found some info!");
			CString message = "File info as follows\n";
			message.appendString(info->isDirectory() ? "the info is a directory" : "The info is a file");
			message.appendString("\nThe info is on the following path : ");
			message.appendString(info->getFullPath());
			CTrace::trace(message, " ");
		}

		// Check its valid
		if (info && info->isDirectory())
		{
			CTrace::trace(info->getFullPath(), "We are opening in the browser = ");
			// Update the selector
			m_selector->setBankPath(info->getFullPath());
			m_selector->update();
			break;
		}
	}
}

//  ===========================================================================
void CPresetBrowser::setMenuWindowAttributes(CWindowAttributes *windowAttributes, const SMenuColours &colours)
{
	m_presetHead->setMenuWindowAttributes(windowAttributes, colours);
}

//  ===========================================================================
void CPresetBrowser::handleActionEvent(const CActionEvent &event)
{
	m_selector->setBankPath(m_presetHead->getBankPath());
}

//  ===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CPresetBrowser::CPresetBrowserSelector, CControlPanel);

//	===========================================================================
CPresetBrowser::CPresetBrowserSelector::CPresetBrowserSelector(IControlRoot *root, const long uniqueId, const CRect &area, CFont *font, IActionListener *listener, IPresetHandler *handler) 
			   :CControlPanel(root->getParentWindow(), root, uniqueId, area, listener)
{
	EXPONENT_CLASS_CONSTRUCTION(CPresetBrowser::CPresetBrowserSelector);
	m_presetHandler = handler;
	m_pageIndex		= 0;
	m_theFont       = font;
	this->onDrawDrawThePanelBounds(false);
}

//	===========================================================================
CPresetBrowser::CPresetBrowserSelector::~CPresetBrowserSelector()
{
	EXPONENT_CLASS_DESTRUCTION(CPresetBrowser::CPresetBrowserSelector);
}

//	===========================================================================
void CPresetBrowser::CPresetBrowserSelector::handleActionEvent(const CActionEvent &event)
{
	// Get the id and the label
	const long id = event.getControl()->getUniqueId();
	CPresetTextLabel *label = (CPresetTextLabel *)event.getControl();

	// Store the string
	const CString text = label->getText();

	// Create the string
	m_filePath = m_selectedPath;
	m_filePath.appendPath(text);
	if (m_presetHandler)
	{
		CString theFilter;
		CString description;
		m_presetHandler->getPresetExtensionFilter(theFilter, description);
		m_filePath.appendNewExtension(theFilter);
	}

	if (label->getSelectionState() == 0)
	{
		// Selected
		for (long i = 0; i < this->getNumberOfControls(); i++)
		{
			IControl *control = this->getControlAtIndex(i);
			if (control && control->getUniqueId() != id)// && control->getUniqueId() != e_nextButton && control->getUniqueId() != e_previousButton)
			{
				((CPresetTextLabel *)control)->selectLabel(false);
			}
		}
	}
	else
	{
		// Selected and load
		CFileInformation *info = m_bankFileList.elementAtIndex(id);
		if (info && m_presetHandler)
		{
			if (!m_presetHandler->loadPreset(info->getFullPath()))
			{
				CDialog::notifyUser("Failed to load preset", "Error!", false);
			}
		}
	}
}

//	===========================================================================
void CPresetBrowser::CPresetBrowserSelector::setBankPath(const CSystemString &bankPath)
{
	// Store the path
	m_selectedPath = bankPath;

	// Clear all the control that we currently have
	this->clearControls();

	// Add the buttons
	//this->addControl(m_nextButton);
	//this->addControl(m_previousButton);

	// Now get the file list
	m_bankFileList.clearArray();
	CFileManager::getFilesInDirectory(m_bankFileList, bankPath);

	// Check we actually have some files
	if (m_bankFileList.getArraySize() == 0)
	{
		return;
	}

	for (long i = 0; i < m_bankFileList.getArraySize(); i++)
	{
		CFileInformation *info = m_bankFileList.elementAtIndex(i);
		if (info)
		{
			CTrace::trace(info->getFullPath(), "Found file ");
			CTrace::trace("---------- FILE INFO ---------- ");
			CTrace::trace(info->getCreationTime().getDayYearString(), "Creation = ");
			CTrace::trace(info->getLastAccessTime().getDayYearString(), "Access = ");
			CTrace::trace(info->getLastWriteTime().getDayYearString(), "Write = ");
			CTrace::trace("-------- END FILE INFO ----------");
		}
	}

	// Page index back to zero
	m_pageIndex = 0;

	// Total font size
	const long fontSize = m_theFont->getFontSize() * 2;

	// Compute the number of elements that we get in a given height (inc space for next and prev buttons
	const long elementsForHeight = (m_area.getHeight() - 14) / fontSize;

	// COmpute the number of elements that we get per the width
	const long elementsForWidth = (long)(m_area.getWidth() / 140.0);

	// Compute the number of elements per page
	const long pageElements = elementsForHeight * elementsForWidth;

	// Placement positions
	long left = 3;
	long top  = 3;
	bool shouldContinue = true;

	// Loop through the files
	for (long i = 0; i < m_bankFileList.getArraySize() && i < pageElements && shouldContinue; i++)
	{
		// Get the information about the file
		CFileInformation *info = m_bankFileList.elementAtIndex(i);

		if (info == NULL)
		{
			continue;
		}

		// Check if we have a preset handler
		if (m_presetHandler == NULL)
		{
			continue;
		}

		// Check the extension
		if (!m_presetHandler->validatePreset(info->getFullPath()))
		{
			continue;
		}

		// Validate that we should be adding this control
		CPresetIO presetIO;

		// Check if its a valid preset
		if (!m_presetHandler->validatePreset(info->getFullPath(), presetIO))
		{
			continue;
		}

		// Close the presetIO
		presetIO.close();

		// Work out the position
		const CRect theArea(left, top, 140, fontSize);

		// Increment the top
		top += fontSize;
		if (top >= (m_area.getHeight() - 14) - fontSize)
		{
			top  =  3;
			left += 140;
		}

		// Get the name of the file
		CSystemString filename = info->getFilename();
		filename.removeExtension();

		// Create the label
		CPresetTextLabel *label = new CPresetTextLabel(this, i, theArea, filename);

		// Set the font and the colour
		label->setFont(m_theFont);
		label->setColours(CAlphaColour::CALPHACOLOUR_BLACK, CAlphaColour::CALPHACOLOUR_BLACK, m_fontColour);
		label->registerActionListener(this);

		// Add the control
		this->addControl(label);

		// Check if we have gone far enough
		if (left + 140 > m_area.getWidth())
		{
			shouldContinue = false;
		}
	}

	// REdraw
	this->update();
}

//  ===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CPresetBrowser::CPresetBrowserSelector::CPresetTextLabel, CTextLabel);

//	===========================================================================
CPresetBrowser::CPresetBrowserSelector::CPresetTextLabel::CPresetTextLabel(IControlRoot *root, const long uniqueId, const CRect &area, const CString &text)
			   :CTextLabel(root, uniqueId, area, text)
{
	EXPONENT_CLASS_CONSTRUCTION(CPresetBrowser::CPresetBrowserSelector::CPresetTextLabel);
	m_selected = false;
	m_load	   = false;
	this->controlIsMouseEnabled();
}

//	===========================================================================
CPresetBrowser::CPresetBrowserSelector::CPresetTextLabel::~CPresetTextLabel()
{
	EXPONENT_CLASS_DESTRUCTION(CPresetBrowser::CPresetBrowserSelector::CPresetTextLabel);
}

//	===========================================================================
void CPresetBrowser::CPresetBrowserSelector::CPresetTextLabel::drawControl(CGraphics &graphics)
{
	// Draw main control
	CTextLabel::drawControl(graphics);
	
	// First check if we can allow the standard handler to draw the disabled control
	if (!this->drawEnabledControl(graphics))
	{
		return;
	}

	// If we are selected draw the control in
	if (m_selected)
	{
		graphics.getMutablePen()->setColour(CAlphaColour::CALPHACOLOUR_ORANGE);
		graphics.drawRectangle(this->getNormalisedArea());
	}
}

//	===========================================================================
void CPresetBrowser::CPresetBrowserSelector::CPresetTextLabel::handleLeftButtonDown(CMouseEvent &event)
{
	// Selected
	m_selected = true;

	// Notify listener
	if (m_actionListener)
	{
		m_actionListener->handleActionEvent(CActionEvent(this, event));
	}

	// Update
	this->update();
}

//	===========================================================================
void CPresetBrowser::CPresetBrowserSelector::CPresetTextLabel::handleDoubleClick(CMouseEvent &event)
{
	// Select and load
	m_selected = true;
	m_load	   = true;

	// Notify listener
	if (m_actionListener)
	{
		m_actionListener->handleActionEvent(CActionEvent(this, event));
	}

	// Update the control
	this->update();
}

//	===========================================================================
void CPresetBrowser::CPresetBrowserSelector::CPresetTextLabel::selectLabel(const bool selected)
{
	m_selected = selected;
	m_load     = false;
	this->update();
}

//	===========================================================================
long CPresetBrowser::CPresetBrowserSelector::CPresetTextLabel::getSelectionState() const
{
	if (m_selected && !m_load)
	{
		return 0;
	}
	else if (m_selected && m_load)
	{
		return 1;
	}
	return -1;
}

//  ===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CPresetBrowser::CPresetBrowserHead, CControlPanel);

//	===========================================================================
CPresetBrowser::CPresetBrowserHead::CPresetBrowserHead(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener, IPresetHandler *presetHandler, CPresetBrowserSelector *selector) 
			   :CControlPanel(root->getParentWindow(), root, uniqueId, area, listener)
{
	EXPONENT_CLASS_CONSTRUCTION(CPresetBrowser::CPresetBrowserHead);
	m_presetHandler = presetHandler;
	m_selector		= selector;
	this->onDrawDrawThePanelBounds(false);
}

//	===========================================================================
CPresetBrowser::CPresetBrowserHead::~CPresetBrowserHead()
{
	EXPONENT_CLASS_DESTRUCTION(CPresetBrowser::CPresetBrowserHead);
	FORGET_COUNTED_OBJECT(m_bankCombo);
	FORGET_COUNTED_OBJECT(m_loadButton);
	FORGET_COUNTED_OBJECT(m_saveButton);
	FORGET_COUNTED_OBJECT(m_bankMenu);
}

//	===========================================================================
void CPresetBrowser::CPresetBrowserHead::intialise(const SPresetBrowserHeadSetup &setup)
{
	// Sotre the bank path
	m_rootPath = m_bankPath = setup.m_rootPath;

	// Create the menu 
	m_bankMenu = new CMenu();
	
	// Create the menu (Banks cannot be nested)
	this->createBankMenu();

	// Create the bank combo
	m_bankCombo = new CComboBox(this, e_bankCombo, setup.m_bankComboRect, CString::CSTRING_EMPTY_STRING, m_bankMenu, this);
	m_bankCombo->setSelectedIndex(0);

	// Create the buttons
	m_loadButton	 = new CMomentaryButton(this, e_loadButton,		setup.m_loadButtonRect,		this);
	m_saveButton	 = new CMomentaryButton(this, e_saveButton,		setup.m_saveButtonRect,		this);

	// Now reference all controls
	m_bankMenu->referenced();
	m_bankCombo->referenced();
	m_loadButton->referenced();
	m_saveButton->referenced();

	// Store the images
	m_bankCombo->setPrimaryImage(setup.m_bankComboUpImage);
	m_bankCombo->setDownImage(setup.m_bankComboDownImage);
	m_loadButton->setPrimaryImage(setup.m_loadButtonUpImage);
	m_loadButton->setDownImage(setup.m_loadButtonDownImage);
	m_saveButton->setPrimaryImage(setup.m_saveButtonUpImage);
	m_saveButton->setDownImage(setup.m_saveButtonDownImage);

	m_bankCombo->setFont(setup.m_theFont);
	m_bankCombo->setColours(CAlphaColour::CALPHACOLOUR_BLACK, CAlphaColour::CALPHACOLOUR_BLACK, setup.m_textColour);

	// Now add all the controls
	this->addControl(m_bankCombo);
	this->addControl(m_loadButton);
	this->addControl(m_saveButton);
}

//	===========================================================================
void CPresetBrowser::CPresetBrowserHead::handleActionEvent(const CActionEvent &event)
{
	switch(event.getControl()->getUniqueId())
	{
		case e_bankCombo:
			{
				CDiskPathMenuItem *item = (CDiskPathMenuItem *)m_bankCombo->getMenuItem(m_bankCombo->getSelectedIndex());
				if (item && m_actionListener)
				{
					m_bankPath = item->getPath();
					m_actionListener->handleActionEvent(CActionEvent(this));
				}
			}
		break;
		case e_loadButton:
			if (m_selector && m_presetHandler)
			{
				// Get the file filter
				CString theFilter;
				CString theExtensionDescription;
				m_presetHandler->getPresetExtensionFilter(theFilter, theExtensionDescription);

				// Opena  file dialog
				CSystemString filename;
				if (CDialog::openFileDialog(m_bankPath, filename, "Load Preset", theExtensionDescription, theFilter))
				{
					if (!m_presetHandler->loadPreset(filename))
					{
						CDialog::notifyUser("Failed to load preset", "Error!");
					}
				}
			}
		break;
		case e_saveButton:
			if (m_presetHandler)
			{
				// Get the file filter
				CString theFilter;
				CString theExtensionDescription;
				m_presetHandler->getPresetExtensionFilter(theFilter, theExtensionDescription);

				// Opena  file dialog
				CSystemString filename;
				if (CDialog::openSaveDialog(m_bankPath, filename, "Save Preset", theExtensionDescription, theFilter))
				{
					bool valid = true;

					// Check if the files exists already and should we overwrite?
					if (CFileManager::doesFileExist(filename))
					{
						if (!CDialog::promptUser("This file already exists, do you want to overwrite?", "Overwrite"))
						{
							valid = false;
						}
					}

					// If they said to overwrite..
					if (valid)
					{
						// Check the filter format
						if (theFilter[0] == '*' && theFilter[1] == '.')
						{
							char theString[10] = "\0";
							char *stringPointer = &theString[0];
							theFilter.getString(theString, 10);
							stringPointer += 2 * sizeof(char);
							theFilter.setString(stringPointer);
						}

						// Check if we have the proper extension
						if (!filename.hasExtension(theFilter))
						{
							filename.appendNewExtension(theFilter, false);
						}

						// Save the preset
						if (!m_presetHandler->savePreset(filename))
						{
							CDialog::notifyUser("Failed to save preset", "Error!");
						}

						// Update the selector
						if (m_selector)
						{
							m_selector->setBankPath(m_bankPath);
						}

						// Create a new bank menu
						this->createBankMenu();

						// Set the mneu window
						m_bankCombo->getMenuWindow()->setMenuAndSize(m_bankMenu);
					}
				}
			}
		break;
	}
}

//	===========================================================================
void CPresetBrowser::CPresetBrowserHead::createBankMenu()
{
	// Get the files to add
	TCountedPointerArray<CFileInformation> files;
	CFileManager::getFilesInDirectory(files, m_rootPath);

	// Clear the bank
	m_bankMenu->clearAllMenuItems();

	// Loop through the files
	long theIndex	    = 0;
	for (long i = 0; i < files.getArraySize(); i++)
	{
		CFileInformation *info = files.elementAtIndex(i);
		if (info && info->isDirectory())
		{
			m_bankMenu->addMenuItem(new CDiskPathMenuItem(info->getFullPath(), info->getFilename(), theIndex++, CMenuItem::e_menuItemNormal)); 
		}
	}
}

//	===========================================================================
void CPresetBrowser::CPresetBrowserHead::setMenuWindowAttributes(CWindowAttributes *windowAttributes, const SMenuColours &colours)
{
	m_bankCombo->setMenuWindowAttributes(windowAttributes, false);
	m_bankCombo->setMenuColours(colours);
}